//-----------------------------------------------------------------------------
// File: XBSound.h
//
// Desc: Helper class for reading a .wav file and playing it in a DirectSound
//       buffer.
//
// Hist: 12.15.00 - New for December XDK release
//       02.15.01 - Updated for March XDK release
//
// Copyright (c) 2000-2001 Microsoft Corporation. All rights reserved.
//-----------------------------------------------------------------------------
#ifndef XBSOUNDEX_H
#define XBSOUNDEX_H
#include <xtl.h>




//-----------------------------------------------------------------------------
// Name: RIFFHEADER
// Desc: For parsing WAV files
//-----------------------------------------------------------------------------
struct RIFFHEADEREX
{
    FOURCC  fccChunkId;
    DWORD   dwDataSize;
};

#define RIFFCHUNK_FLAGS_VALID   0x00000001

typedef struct localfilehandle_S {
	unsigned char *data;
	unsigned long size;
	unsigned long pos;
} localfilehandle;



//-----------------------------------------------------------------------------
// Name: class CRiffChunk
// Desc: RIFF chunk utility class
//-----------------------------------------------------------------------------
class CRiffChunkEx
{
    FOURCC            m_fccChunkId;       // Chunk identifier
    const CRiffChunkEx* m_pParentChunk;     // Parent chunk
    localfilehandle * m_file;
    DWORD             m_dwDataOffset;     // Chunk data offset
    DWORD             m_dwDataSize;       // Chunk data size
    DWORD             m_dwFlags;          // Chunk flags

public:
    CRiffChunkEx();

    // Initialization
    VOID    Initialize( FOURCC fccChunkId, const CRiffChunkEx* pParentChunk, localfilehandle* file );
    HRESULT Open();
    BOOL    IsValid()     { return !!(m_dwFlags & RIFFCHUNK_FLAGS_VALID); }

    // Data
    HRESULT ReadData( LONG lOffset, VOID* pData, DWORD dwDataSize );

    // Chunk information
    FOURCC  GetChunkId()  { return m_fccChunkId; }
    DWORD   GetDataSize() { return m_dwDataSize; }
};




//-----------------------------------------------------------------------------
// Name: class CWaveFile
// Desc: Wave file utility class
//-----------------------------------------------------------------------------
class CWaveFileEx
{
    localfilehandle m_file;
    CRiffChunkEx  m_RiffChunk;        // RIFF chunk
    CRiffChunkEx  m_FormatChunk;      // Format chunk
    CRiffChunkEx  m_DataChunk;        // Data chunk
    CRiffChunkEx  m_WaveSampleChunk;  // Wave Sample chunk
    
public:
    CWaveFileEx();
    ~CWaveFileEx();

    // Initialization
    HRESULT Open( const CHAR* strFileName );
    VOID    Close();

    // File format
    HRESULT GetFormat( WAVEFORMATEX* pwfxFormat, DWORD dwFormatSize, DWORD *pdwRequiredSize = NULL );

    // File data
    HRESULT ReadSample( DWORD dwPosition, VOID* pBuffer, DWORD dwBufferSize, 
                        DWORD* pdwRead );

    // Loop region
    HRESULT GetLoopRegion( DWORD* pdwStart, DWORD* pdwLength );

    // File properties
    VOID    GetDuration( DWORD* pdwDuration ) { *pdwDuration = m_DataChunk.GetDataSize(); }
};


//-----------------------------------------------------------------------------
// Name: class CSound
// Desc: Encapsulates functionality of a DirectSound buffer.
//-----------------------------------------------------------------------------
class CXBSoundEx
{
protected:
    LPDIRECTSOUNDBUFFER  m_pDSoundBuffer;
    WAVEFORMATEX         m_WaveFormat;
    DSBUFFERDESC         m_dsbd;
    DWORD                m_dwBufferSize;

public:
    HRESULT Create( const CHAR* strFileName, DWORD dwFlags = 0L );
    HRESULT Create( const WAVEFORMATEX* pwfxFormat, DWORD dwFlags,
                    const VOID* pBuffer, DWORD dwBytes );
    VOID    Destroy();

    HRESULT Play( DWORD dwFlags = 0L ) const;
    HRESULT Stop() const;
    HRESULT SetPosition( const D3DXVECTOR3& vPosition ) const;
    HRESULT SetVelocity( const D3DXVECTOR3& vVelocity ) const;
	HRESULT SetVolume( const long lVolume ) const;

    CXBSoundEx();
    ~CXBSoundEx();
};


#endif // XBSOUND_H
